'use client';
import type { InventoryData } from '@/components/dashboard/dashboard';

export type ExtendedInventoryData = InventoryData & {
  Revenue: number;
  Profit: number;
  'Cost of Goods Sold': number;
  'Profit Margin (%)': number;
  'Gross Margin (%)': number;
  'Stock Turnover Ratio': number;
  'Stock Movement': number;
  Category: string;
};

function sanitizeNumber(num: number): number {
  if (isNaN(num) || !isFinite(num)) {
    return 0;
  }
  return num;
}


function enhanceRow(item: InventoryData): ExtendedInventoryData {
  const sellingPrice = (item['Selling Price in INR'] as number) || 0;
  const costPrice = (item['Cost Price in INR'] as number) || 0;
  const salesUnits = (item['Sales (Units)'] as number) || 0;
  const openingStock = (item['Opening Stock (Units)'] as number) || 0;
  const purchaseUnits = (item['Purchase (Units)'] as number) || 0;
  const closingStock = (item['Closing Stock (Units)'] as number) || 0;

  const revenue = sellingPrice * salesUnits;
  const cogs = costPrice * salesUnits;
  const profit = revenue - cogs;

  const profitMargin = revenue > 0 ? (profit / revenue) * 100 : 0;
  const grossMargin = revenue > 0 ? ((revenue - cogs) / revenue) * 100 : 0;

  const avgInventory = (openingStock + closingStock) / 2;
  const stockTurnoverRatio = avgInventory > 0 ? salesUnits / avgInventory : 0;
  
  const totalStockAvailable = openingStock + purchaseUnits;
  const stockMovement = totalStockAvailable > 0 ? (salesUnits / totalStockAvailable) * 100 : 0;

  const itemName = (item['Item Name'] as string) || 'Unknown';
  const category = itemName.split(' ')[0];


  return {
    ...item,
    Revenue: sanitizeNumber(revenue),
    'Cost of Goods Sold': sanitizeNumber(cogs),
    Profit: sanitizeNumber(profit),
    'Profit Margin (%)': sanitizeNumber(profitMargin),
    'Gross Margin (%)': sanitizeNumber(grossMargin),
    'Stock Turnover Ratio': sanitizeNumber(stockTurnoverRatio),
    'Stock Movement': sanitizeNumber(stockMovement),
    Category: category,
  };
}

export const calculateClosingStockValue = (
  data: InventoryData[],
  type: 'cost' | 'selling'
): number => {
  const priceKey =
    type === 'cost' ? 'Cost Price in INR' : 'Selling Price in INR';
  return data.reduce((total, item) => {
    const closingStock = Number(item['Closing Stock (Units)']) || 0;
    const price = Number(item[priceKey]) || 0;
    return total + closingStock * price;
  }, 0);
};

export const calculatePotentialProfit = (data: InventoryData[]): number => {
  return data.reduce((total, item) => {
    const closingStock = Number(item['Closing Stock (Units)']) || 0;
    const sellingPrice = Number(item['Selling Price in INR']) || 0;
    const costPrice = Number(item['Cost Price in INR']) || 0;
    if (sellingPrice > costPrice) {
      return total + (sellingPrice - costPrice) * closingStock;
    }
    return total;
  }, 0);
};

export const getReorderAlerts = (data: InventoryData[]): InventoryData[] => {
  return data.filter((item) => {
    const closingStock = (item['Closing Stock (Units)'] as number) || 0;
    const reorderLevel = (item['Reorder Level'] as number) || 0;
    return reorderLevel > 0 && closingStock < reorderLevel;
  });
};

export const getTopProductsBySales = (
  data: InventoryData[],
  count: number
): InventoryData[] => {
  return [...data]
    .sort(
      (a, b) =>
        (b['Sales (Units)'] as number) - (a['Sales (Units)'] as number)
    )
    .slice(0, count);
};

export const getTopProductsByRevenue = (
  data: InventoryData[],
  count: number
): InventoryData[] => {
  return [...data]
    .map((item) => ({
      ...item,
      Revenue:
        ((item['Sales (Units)'] as number) || 0) *
        ((item['Selling Price in INR'] as number) || 0),
    }))
    .sort((a, b) => (b.Revenue as number) - (a.Revenue as number))
    .slice(0, count);
};

export const calculateProductMetrics = (data: InventoryData[]) => {
  return data.map((item) => {
    const sellingPrice = (item['Selling Price in INR'] as number) || 0;
    const costPrice = (item['Cost Price in INR'] as number) || 0;
    const salesUnits = (item['Sales (Units)'] as number) || 0;

    const profit = (sellingPrice - costPrice) * salesUnits;
    const profitMargin =
      sellingPrice > 0 ? ((sellingPrice - costPrice) / sellingPrice) * 100 : 0;

    return {
      ...item,
      'Product-wise Profit': sanitizeNumber(profit),
      'Profit Margin (%)': sanitizeNumber(profitMargin),
    };
  });
};

export const calculateStockMovement = (data: InventoryData[]): Record<string, number> => {
  const stockMovement: Record<string, number> = {};
  data.forEach((item) => {
    const sales = (item['Sales (Units)'] as number) || 0;
    const opening = (item['Opening Stock (Units)'] as number) || 0;
    const purchase = (item['Purchase (Units)'] as number) || 0;
    const totalStock = opening + purchase;
    const movement = totalStock > 0 ? (sales / totalStock) * 100 : 0;
    stockMovement[item.SKU as string] = sanitizeNumber(movement);
  });
  return stockMovement;
}

export const getFastVsSlowMovers = (data: InventoryData[]) => {
  const turnoverData = data.map(item => {
    const openingStock = (item['Opening Stock (Units)'] as number) || 0;
    const closingStock = (item['Closing Stock (Units)'] as number) || 0;
    const sales = (item['Sales (Units)'] as number) || 0;
    
    const avgInventory = (openingStock + closingStock) / 2;
    const turnoverRatio = avgInventory > 0 ? sales / avgInventory : 0;

    return { ...item, TurnoverRatio: sanitizeNumber(turnoverRatio) };
  }).sort((a, b) => b.TurnoverRatio - a.TurnoverRatio);

  const fastMovers = turnoverData.filter(p => p.TurnoverRatio > 0);
  const slowMovers = [...turnoverData].sort((a, b) => a.TurnoverRatio - b.TurnoverRatio).filter(p => p.TurnoverRatio > 0);

  return { fastMovers, slowMovers };
}

export const aggregateInventoryData = (
  data: InventoryData[],
  dimension: 'Item Name' | 'SKU' | 'Category',
  metric:
    | 'Revenue'
    | 'Profit'
    | 'Sales (Units)'
    | 'Closing Stock (Units)'
    | 'Opening Stock (Units)'
    | 'Purchase (Units)'
    | 'Cost of Goods Sold'
    | 'Profit Margin (%)'
    | 'Stock Turnover Ratio'
    | 'Reorder Level',
  returnFullData = false
): { name: string; value: number }[] | ExtendedInventoryData[] => {
  
  const enhancedData = data.map(enhanceRow);
  
  if (returnFullData) {
    return enhancedData;
  }
  
  const aggregated: Record<string, number> = {};
  const tempAggregations: Record<string, { totalRevenue: number; totalProfit: number; totalSales: number; totalCogs: number; avgInventory: number; count: number; }> = {};


  enhancedData.forEach((item) => {
    const key = item[dimension] as string;
    if (!key) return;

    if (!aggregated[key]) {
      aggregated[key] = 0;
    }
    
    // For metrics that need to be aggregated (summed up)
    const value_to_add = item[metric as keyof ExtendedInventoryData] as number || 0;
    
    if (metric !== 'Profit Margin (%)' && metric !== 'Stock Turnover Ratio') {
      aggregated[key] += value_to_add;
    }
  });
  
  // For ratio metrics, we need to recalculate them on the aggregated data
  if (metric === 'Profit Margin (%)' || metric === 'Stock Turnover Ratio') {
     const grouped: Record<string, ExtendedInventoryData[]> = {};
     enhancedData.forEach(item => {
        const key = item[dimension] as string;
        if (!grouped[key]) {
            grouped[key] = [];
        }
        grouped[key].push(item);
     });

     for(const key in grouped) {
        const group = grouped[key];
        const totalRevenue = group.reduce((sum, item) => sum + item.Revenue, 0);
        const totalProfit = group.reduce((sum, item) => sum + item.Profit, 0);
        const totalSalesUnits = group.reduce((sum, item) => sum + (item['Sales (Units)'] as number), 0);
        const totalOpeningStock = group.reduce((sum, item) => sum + (item['Opening Stock (Units)'] as number), 0);
        const totalClosingStock = group.reduce((sum, item) => sum + (item['Closing Stock (Units)'] as number), 0);

        const avgInventory = (totalOpeningStock + totalClosingStock) / 2;

        if (metric === 'Profit Margin (%)') {
            aggregated[key] = totalRevenue > 0 ? (totalProfit / totalRevenue) * 100 : 0;
        }
        if (metric === 'Stock Turnover Ratio') {
            aggregated[key] = avgInventory > 0 ? totalSalesUnits / avgInventory : 0;
        }
        aggregated[key] = sanitizeNumber(aggregated[key]);
     }
  }


  return Object.entries(aggregated)
    .map(([name, value]) => ({ name, value }))
    .sort((a, b) => b.value - a.value);
};


export const getProfitableReorderItems = (data: InventoryData[]) => {
  const reorderAlerts = getReorderAlerts(data);
  const productMetrics = calculateProductMetrics(reorderAlerts);

  return productMetrics
    .filter(item => item['Product-wise Profit'] > 0)
    .sort((a, b) => b['Product-wise Profit'] - a['Product-wise Profit']);
}
